const DEFAULT_BASE_URL = "https://myamproject.my.id";
const DASHBOARD_PATH = "/index.html";
const MEMBER_VIEWS = new Set(["dashboard", "purchases", "tutorial", "request"]);
const DASHBOARD_HOSTS = new Set(["localhost", "127.0.0.1"]);

const EXT_KEYS = {
  token: "dxp_token",
  baseUrl: "dx_ext_base_url",
  user: "dxp_user",
  pendingView: "dx_ext_pending_view",
  managedCookies: "dx_ext_managed_cookies"
};
const THEME_KEY = "dx_ext_theme";


const state = {
  baseUrl: DEFAULT_BASE_URL,
  token: "",
  dashboardTabId: null,
  services: [],
  currentFolder: null, // Base name of the current folder being viewed
};

const els = {};
let toastTimer = null;

function $(id) {
  return document.getElementById(id);
}

function setHidden(element, hidden) {
  if (!element) return;
  element.classList.toggle("hidden", hidden);
}

function setStatus(type, text, hint) {
  if (!els.statusIndicator || !els.statusText || !els.statusHint) return;
  els.statusIndicator.className = `status-dot ${type}`;
  els.statusText.textContent = text;
  els.statusHint.textContent = hint;
}


function setLoading(isLoading, text = "Memproses...") {
  if (!els.loadingOverlay || !els.loadingText) return;
  els.loadingText.textContent = text;
  setHidden(els.loadingOverlay, !isLoading);
}

function applyTheme(theme) {
  const next = theme === "dark" ? "dark" : "light";
  document.body.setAttribute("data-theme", next);
  if (els.themeToggle) els.themeToggle.textContent = next === "light" ? "☀️" : "🌙";
  localStorage.setItem(THEME_KEY, next);
}

function toggleTheme() {
  const current = localStorage.getItem(THEME_KEY) || "light";
  applyTheme(current === "light" ? "dark" : "light");
}

function showToast(message, type = "success") {
  if (!els.toast) return;
  els.toast.textContent = message;
  els.toast.className = `toast ${type}`;
  setHidden(els.toast, false);

  if (toastTimer) clearTimeout(toastTimer);
  toastTimer = setTimeout(() => {
    setHidden(els.toast, true);
    els.toast.className = "toast hidden";
  }, 2800);
}

function safeParseUrl(rawUrl) {
  try {
    return new URL(rawUrl);
  } catch (_) {
    return null;
  }
}

function isDashboardUrl(rawUrl) {
  const parsed = safeParseUrl(rawUrl);
  if (!parsed) return false;

  const hostname = parsed.hostname.toLowerCase();

  const baseParsed = safeParseUrl(state.baseUrl || DEFAULT_BASE_URL);
  if (baseParsed && hostname === baseParsed.hostname.toLowerCase()) return true;

  // Match localhost, 127.0.0.1 or any .myamproject.id domain
  if (DASHBOARD_HOSTS.has(hostname)) return true;
  if (hostname === "localhost" || hostname === "127.0.0.1") return true;
  if (/^192\.168\.\d+\.\d+$/.test(hostname) || /^10\.\d+\.\d+\.\d+$/.test(hostname)) return true;

  return hostname.endsWith(".myamproject.my.id") || hostname === "myamproject.my.id";
}


function deriveBaseUrlFromTab(tab) {
  const parsed = safeParseUrl(tab?.url || "");
  return parsed?.origin || DEFAULT_BASE_URL;
}

function queryTabs(queryInfo) {
  return new Promise((resolve) => {
    chrome.tabs.query(queryInfo, (tabs) => resolve(tabs || []));
  });
}

function sendTabMessage(tabId, payload) {
  return new Promise((resolve) => {
    try {
      chrome.tabs.sendMessage(tabId, payload, (response) => {
        if (chrome.runtime.lastError) {
          resolve(null);
          return;
        }
        resolve(response || null);
      });
    } catch (_) {
      resolve(null);
    }
  });
}

function updateTab(tabId, updateProperties) {
  return new Promise((resolve) => {
    chrome.tabs.update(tabId, updateProperties, (tab) => resolve(tab || null));
  });
}

function createTab(createProperties) {
  return new Promise((resolve) => {
    chrome.tabs.create(createProperties, (tab) => resolve(tab || null));
  });
}

function reloadTab(tabId) {
  return new Promise((resolve) => {
    chrome.tabs.reload(tabId, {}, () => resolve());
  });
}

function storageGet(keys) {
  return new Promise((resolve) => {
    chrome.storage.local.get(keys, (result) => resolve(result || {}));
  });
}

function storageSet(data) {
  return new Promise((resolve) => {
    chrome.storage.local.set(data, () => resolve());
  });
}

function storageRemove(keys) {
  return new Promise((resolve) => {
    chrome.storage.local.remove(keys, () => resolve());
  });
}

async function findDashboardTab() {
  const currentWindowTabs = await queryTabs({ currentWindow: true });
  const currentMatch = currentWindowTabs.find((tab) => isDashboardUrl(tab.url || ""));
  if (currentMatch) return currentMatch;

  const allTabs = await queryTabs({});
  return allTabs.find((tab) => isDashboardUrl(tab.url || "")) || null;
}

async function readStoredSession() {
  const result = await storageGet([EXT_KEYS.token, EXT_KEYS.baseUrl, EXT_KEYS.user]);
  let user = null;
  try {
    user = result[EXT_KEYS.user] ? JSON.parse(result[EXT_KEYS.user]) : null;
  } catch (e) { }

  return {
    token: result[EXT_KEYS.token] || "",
    baseUrl: result[EXT_KEYS.baseUrl] || DEFAULT_BASE_URL,
    user: user
  };
}


async function saveManagedCookie(url, name) {
  const result = await storageGet([EXT_KEYS.managedCookies]);
  const cookies = result[EXT_KEYS.managedCookies] || [];

  const exists = cookies.some(c => c.url === url && c.name === name);
  if (!exists) {
    cookies.push({ url, name });
    await storageSet({ [EXT_KEYS.managedCookies]: cookies });
  }
}

async function clearManagedCookies() {
  const result = await storageGet([EXT_KEYS.managedCookies]);
  const cookies = result[EXT_KEYS.managedCookies] || [];

  if (cookies.length === 0) return;

  for (const cookie of cookies) {
    await new Promise((resolve) => {
      chrome.cookies.remove({ url: cookie.url, name: cookie.name }, () => resolve());
    });
  }

  await storageRemove([EXT_KEYS.managedCookies]);
  console.log(`Cleared ${cookies.length} managed cookies.`);
}

async function clearStoredSession() {
  await storageRemove([EXT_KEYS.token, EXT_KEYS.baseUrl, EXT_KEYS.email, EXT_KEYS.name, EXT_KEYS.pendingView]);
}

async function syncSessionFromDashboardTab() {
  const tab = await findDashboardTab();
  if (!tab) return null;

  state.dashboardTabId = tab.id;
  const fallbackBaseUrl = deriveBaseUrlFromTab(tab);
  const payload = await sendTabMessage(tab.id, { action: "myam_project_get_session" });
  if (!payload) {
    throw new Error("TAB_NOT_RESPONDING");
  }

  const synced = {
    tabId: tab.id,
    token: payload.token || "",
    user: payload.user || null,
    baseUrl: payload.baseUrl || fallbackBaseUrl
  };

  if (synced.token) {
    state.token = synced.token;
    state.baseUrl = synced.baseUrl;
    await storageSet({
      [EXT_KEYS.token]: synced.token,
      [EXT_KEYS.baseUrl]: synced.baseUrl,
      [EXT_KEYS.user]: JSON.stringify(synced.user)
    });
  }

  return synced;
}


async function fetchSessionData(baseUrl, token) {
  // 1. Fetch basic session to check status/version/user limits
  const currentManifestVersion = chrome.runtime.getManifest().version;
  const response = await fetch(`${baseUrl}/api/session`, {
    method: "GET",
    headers: {
      "Authorization": `Bearer ${token}`,
      "x-extension-version": currentManifestVersion
    }
  });

  if (response.status === 401) throw new Error("SESSION_EXPIRED");
  if (response.status === 403) {
    const body = await response.json().catch(() => ({}));
    if (body.error === 'SUSPENDED') {
      const err = new Error("SUSPENDED");
      err.suspendMessage = body.message || 'Akun Anda telah ditangguhkan.';
      err.adminContact = body.admin_contact || 'https://t.me/raykram';
      throw err;
    }
    throw new Error(`HTTP_403`);
  }
  if (!response.ok) throw new Error(`HTTP_${response.status}`);

  const payload = await response.json();
  if (!payload.status) throw new Error("SESSION_INVALID");

  // 2. Fetch secure extension data for cookies
  try {
    const extRes = await fetch(`${baseUrl}/api/extension/sync`, {
      method: "GET",
      headers: {
        "Authorization": `Bearer ${token}`,
        "x-extension-request": "true"
      }
    });

    if (extRes.ok) {
      const extData = await extRes.json();
      if (extData.status && extData.items) {
        payload.items = extData.items; // Override with secure items
      }
    }
  } catch (e) {
    console.warn("Gagal secure sync");
  }

  return payload;
}

function deobfuscateCookie(b64) {
  if (!b64) return "";
  try {
    const key = "dxprime_secure_sync_2026";
    const xorStr = atob(b64);
    let original = "";
    for (let i = 0; i < xorStr.length; i++) {
      original += String.fromCharCode(xorStr.charCodeAt(i) ^ key.charCodeAt(i % key.length));
    }
    return original;
  } catch (e) {
    console.warn("Gagal deobfuscate cookie");
    return "";
  }
}

function getFriendlyError(errorCode) {
  if (errorCode === "TAB_NOT_RESPONDING") {
    return "Gagal sinkron. Segarkan halaman dashboard lalu coba lagi.";
  }
  if (errorCode === "SESSION_EXPIRED") {
    return "Sesi kedaluwarsa. Login ulang di dashboard web.";
  }
  if (errorCode === "SESSION_INVALID") {
    return "Sesi dashboard tidak valid.";
  }
  if (errorCode === "DEVICE_LIMIT") {
    return "Batas 2 perangkat tercapai. Logout dari perangkat lain terlebih dahulu.";
  }
  if (typeof errorCode === "string" && errorCode.startsWith("HTTP_")) {
    return `Server mengembalikan ${errorCode.replace("HTTP_", "HTTP ")}.`;
  }
  return "Gagal sinkronisasi sesi dengan dashboard.";
}

function isVersionOlder(current, min) {
  if (!min) return false;
  const currParts = current.split('.').map(Number);
  const minParts = min.split('.').map(Number);
  const len = Math.max(currParts.length, minParts.length);
  for (let i = 0; i < len; i++) {
    const c = currParts[i] || 0;
    const m = minParts[i] || 0;
    if (c < m) return true;
    if (c > m) return false;
  }
  return false;
}

function renderUpdateRequired(minVersion) {
  state.token = "";
  state.services = [];
  clearManagedCookies();

  setStatus("error", "Update Ekstensi", `Versi usang. Minimal versi: ${minVersion}. Buka dashboard untuk unduh versi terbaru.`);
  setHidden(els.authCard, false);
  setHidden(els.userCard, true);
  setHidden(els.controlsCard, true);
  setHidden(els.servicesCard, true);

  if (els.serviceList) els.serviceList.innerHTML = "";
  if (els.servicesCount) els.servicesCount.textContent = "0 layanan";

  if (els.openDashboardBtn) {
    els.openDashboardBtn.style.display = "flex";
    els.openDashboardBtn.textContent = "Buka Dashboard untuk Update";
  }

  if (els.retrySyncBtn) {
    els.retrySyncBtn.style.display = "none";
  }
}

function renderSuspended(message, adminContact) {
  state.token = "";
  state.services = [];

  const contact = adminContact || "https://t.me/raykram";

  setStatus("error", "Akun Ditangguhkan", message || "Hubungi admin untuk bantuan.");
  setHidden(els.authCard, false);
  setHidden(els.userCard, true);
  setHidden(els.controlsCard, true);
  setHidden(els.servicesCard, true);

  if (els.serviceList) els.serviceList.innerHTML = "";

  // Tampilkan tombol kontak admin jika ada elemen retrySyncBtn
  if (els.retrySyncBtn) {
    els.retrySyncBtn.textContent = "📩 Hubungi Admin Telegram";
    els.retrySyncBtn.onclick = () => window.open(contact, "_blank");
  }
  if (els.openDashboardBtn) {
    els.openDashboardBtn.style.display = "none";
  }

  clearManagedCookies();
}

function renderDisconnected(reason) {
  state.token = "";
  state.services = [];

  setStatus("error", "Belum terhubung ke akun", reason || "Login di website MYAM Project lalu sinkron ulang.");
  setHidden(els.authCard, false);
  setHidden(els.userCard, true);
  setHidden(els.controlsCard, true);
  setHidden(els.servicesCard, true);

  if (els.serviceList) els.serviceList.innerHTML = "";
  if (els.servicesCount) els.servicesCount.textContent = "0 layanan";

  clearManagedCookies();
}

function getMemberBadge(role, isPremium) {
  if (role === "admin") return { text: "Admin", className: "admin" };
  if (isPremium) return { text: "Premium", className: "premium" };
  return { text: "Free", className: "free" };
}

function renderSessionInfo(payload, storedFallback = {}) {
  // Payload from /session is the user profile itself
  const role = String(payload.role || payload.user?.role || "user").toLowerCase();
  const isPremium = payload.isPremium === true || payload.status === 'premium' || role === "admin";
  const badge = getMemberBadge(role, isPremium);

  const user = payload.user || storedFallback.user || payload;
  const displayName = user.name || user.email || "Member";
  const displayEmail = user.email || "-";


  if (els.userName) els.userName.textContent = displayName;
  if (els.userEmail) els.userEmail.textContent = displayEmail;

  if (els.memberBadge) {
    els.memberBadge.textContent = badge.text;
    els.memberBadge.className = `badge ${badge.className}`;
  }


  setHidden(els.authCard, true);
  setHidden(els.userCard, false);
  setHidden(els.controlsCard, false);
  setHidden(els.servicesCard, false);

  if (!isPremium && role !== "admin") {
    state.services = [];
    if (els.servicesEmpty) {
      els.servicesEmpty.textContent = "Akun Anda masih Free. Upgrade di dashboard untuk mengakses layanan premium.";
      setHidden(els.servicesEmpty, false);
    }
    if (els.serviceList) els.serviceList.innerHTML = "";
    if (els.servicesCount) els.servicesCount.textContent = "0 layanan";
    setStatus("ready", "Akun terhubung", "Status akun: Free Member.");

    // Auto-clear cookies if they were set from a previous premium session
    clearManagedCookies();
    return;
  }

  state.services = Array.isArray(payload.items) ? payload.items : [];
  setStatus("ready", "Akun terhubung", "Pilih layanan untuk menerapkan cookie otomatis.");
  renderServices();
}

function escapeHtml(raw) {
  return String(raw || "")
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;")
    .replace(/\"/g, "&quot;")
    .replace(/'/g, "&#39;");
}

function getBaseName(name) {
  // Remove trailing numbers (optional space before)
  return (name || "").replace(/\s*\d+$/, "").trim();
}

function renderServices() {
  if (!els.serviceList || !els.servicesEmpty || !els.servicesCount) return;

  const keyword = (els.serviceSearch?.value || "").trim().toLowerCase();
  let itemsToRender = [];

  if (keyword) {
    // If searching, show all matches without grouping
    itemsToRender = state.services
      .filter(item => String(item?.name || "").toLowerCase().includes(keyword))
      .sort((a, b) => String(a.name).localeCompare(String(b.name)));
    state.currentFolder = null;
  } else if (state.currentFolder) {
    // If in a folder, show items belonging to that folder
    itemsToRender = state.services
      .filter(item => getBaseName(item.name) === state.currentFolder)
      .sort((a, b) => String(a.name).localeCompare(String(b.name)));
  } else {
    // Root view: group items
    const groups = new Map();
    state.services.forEach(item => {
      const base = getBaseName(item.name);
      if (!groups.has(base)) groups.set(base, []);
      groups.get(base).push(item);
    });

    const entries = Array.from(groups.entries()).sort((a, b) => a[0].localeCompare(b[0]));

    entries.forEach(([base, members]) => {
      if (members.length > 1) {
        // Render as folder
        itemsToRender.push({
          isFolder: true,
          name: base,
          count: members.length,
          icon: members[0].icon
        });
      } else {
        // Render as single item
        itemsToRender.push(members[0]);
      }
    });
  }

  els.serviceList.innerHTML = "";
  els.servicesCount.textContent = keyword ? `${itemsToRender.length} ditemukan` : (state.currentFolder ? state.currentFolder : `${itemsToRender.length} kategori`);

  if (state.currentFolder && !keyword) {
    const backBtn = document.createElement("button");
    backBtn.className = "service-card back-btn";
    backBtn.innerHTML = `
      <div class="service-top">
        <div class="service-icon back-icon">⬅</div>
        <p class="service-name">Kembali</p>
      </div>
      <p class="service-note">Kembali ke semua layanan</p>
    `;
    backBtn.onclick = () => {
      state.currentFolder = null;
      renderServices();
    };
    els.serviceList.appendChild(backBtn);
  }

  if (itemsToRender.length === 0) {
    els.servicesEmpty.textContent = keyword
      ? "Layanan tidak ditemukan untuk kata kunci tersebut."
      : "Belum ada layanan yang tersedia untuk akun ini.";
    setHidden(els.servicesEmpty, false);
    return;
  }

  setHidden(els.servicesEmpty, true);

  itemsToRender.forEach((item) => {
    if (item.isFolder) {
      renderFolderCard(item);
    } else {
      renderServiceCard(item);
    }
  });
}

function renderFolderCard(folder) {
  const button = document.createElement("button");
  button.type = "button";
  button.className = "service-card folder-card";
  const icon = folder.icon ? escapeHtml(folder.icon) : "icon.png";

  button.innerHTML = `
    <div class="service-top">
      <div class="folder-icon-wrapper">
        <img class="service-icon" src="${icon}" alt="${folder.name}">
        <span class="folder-badge">📁</span>
      </div>
      <p class="service-name">${escapeHtml(folder.name)}</p>
    </div>
    <p class="service-note">${folder.count} Akun Tersedia</p>
  `;

  button.onclick = () => {
    state.currentFolder = folder.name;
    els.serviceSearch.value = ""; // Clear search when opening folder
    renderServices();
  };
  els.serviceList.appendChild(button);
}

function renderServiceCard(item) {
  const cookieData = deobfuscateCookie(item?._sec_data) || item?.cookie_value || item?.value;
  const hasCookie = Boolean(cookieData);
  const button = document.createElement("button");
  button.type = "button";
  button.className = "service-card";
  button.disabled = !hasCookie;

  const icon = item?.icon ? escapeHtml(item.icon) : "icon.png";
  const name = escapeHtml(item?.name || "Layanan");
  const note = hasCookie ? "Klik untuk aktifkan di browser" : "Cookie belum tersedia";

  button.innerHTML = `
    <div class="service-top">
      <img class="service-icon" src="${icon}" alt="${name}">
      <p class="service-name">${name}</p>
    </div>
    <p class="service-note">${escapeHtml(note)}</p>
  `;

  const iconNode = button.querySelector(".service-icon");
  if (iconNode) {
    iconNode.addEventListener("error", () => {
      iconNode.src = "icon.png";
    });
  }

  if (!hasCookie) {
    button.style.opacity = "0.58";
    button.style.cursor = "not-allowed";
    button.style.borderColor = "rgba(239, 68, 68, 0.35)";
  } else {
    button.addEventListener("click", () => activateService(item));
  }

  els.serviceList.appendChild(button);
}

function normalizeSameSite(rawValue) {
  const value = String(rawValue || "").toLowerCase();
  if (value === "strict") return "strict";
  if (value === "lax") return "lax";
  if (value === "none" || value === "no_restriction" || value === "no restriction") return "no_restriction";
  return "unspecified";
}

function getCookieBaseUrl(cookie) {
  const domain = String(cookie?.domain || "").replace(/^\./, "").trim();
  if (!domain) return null;
  return `https://${domain}/`;
}

function setBrowserCookie(cookie, targetUrls) {
  return new Promise((resolve, reject) => {
    const baseUrl = getCookieBaseUrl(cookie);
    if (!baseUrl || !cookie?.name) {
      resolve();
      return;
    }

    const sameSite = normalizeSameSite(cookie.sameSite);
    const secure = Boolean(cookie.secure || sameSite === "no_restriction");
    const now = Math.floor(Date.now() / 1000);
    const expirationDate = Number.isFinite(Number(cookie.expirationDate))
      ? Number(cookie.expirationDate)
      : now + (10 * 24 * 60 * 60);

    const details = {
      url: baseUrl,
      name: String(cookie.name),
      value: String(cookie.value || ""),
      path: String(cookie.path || "/"),
      secure,
      httpOnly: Boolean(cookie.httpOnly),
      sameSite,
      expirationDate
    };

    if (cookie.domain) {
      details.domain = String(cookie.domain);
    }

    chrome.cookies.remove({ url: baseUrl, name: String(cookie.name) }, () => {
      chrome.cookies.set(details, () => {
        if (chrome.runtime.lastError) {
          reject(new Error(chrome.runtime.lastError.message));
          return;
        }
        targetUrls.add(baseUrl);
        saveManagedCookie(baseUrl, String(cookie.name)).then(resolve);
      });
    });
  });
}

async function activateService(item) {
  const cookieData = deobfuscateCookie(item?._sec_data) || item?.cookie_value || item?.value;
  if (!cookieData) {
    showToast("Cookie layanan belum tersedia.", "error");
    return;
  }

  setLoading(true, `Mengaktifkan ${item.name || "layanan"}...`);

  const targetUrls = new Set();
  const failures = [];

  let cookiesToProcess = [];
  let providedUrl = null;

  try {
    const parsed = typeof cookieData === 'string' ? JSON.parse(cookieData) : cookieData;

    // Check for the new complex format: {"url": "...", "cookies": [...]}
    if (parsed && typeof parsed === 'object' && Array.isArray(parsed.cookies)) {
      cookiesToProcess = parsed.cookies;
      providedUrl = parsed.url;
    } else if (Array.isArray(parsed)) {
      cookiesToProcess = parsed;
    } else {
      throw new Error("INVALID_FORMAT");
    }
  } catch (e) {
    showToast(`Format cookie ${item.name || "layanan"} tidak valid.`, "error");
    setLoading(false);
    return;
  }

  if (cookiesToProcess.length === 0) {
    showToast("Data cookie kosong.", "error");
    setLoading(false);
    return;
  }

  for (const cookie of cookiesToProcess) {
    try {
      await setBrowserCookie(cookie, targetUrls);
    } catch (error) {
      failures.push(error.message);
    }
  }

  if (targetUrls.size === 0 && !providedUrl) {
    setLoading(false);
    showToast("Tidak ada domain target valid dari cookie.", "error");
    return;
  }

  // Use providedUrl if available, otherwise use detected target URLs
  if (providedUrl) {
    await createTab({ url: providedUrl });
  } else {
    for (const url of targetUrls) {
      await createTab({ url });
    }
  }

  setLoading(false);

  if (failures.length > 0) {
    showToast(`Sebagian cookie gagal dipasang (${failures.length}).`, "error");
    return;
  }

  showToast(`${item.name || "Layanan"} aktif di ${targetUrls.size} domain.`, "success");
}

async function openDashboard(view = "dashboard") {
  const selectedView = MEMBER_VIEWS.has(view) ? view : "dashboard";

  const stored = await readStoredSession();
  const baseUrl = state.baseUrl || stored.baseUrl || DEFAULT_BASE_URL;
  state.baseUrl = baseUrl;
  const dashboardUrl = `${baseUrl}${DASHBOARD_PATH}`;

  if (selectedView !== "dashboard") {
    await storageSet({ [EXT_KEYS.pendingView]: selectedView });
  } else {
    await storageRemove([EXT_KEYS.pendingView]);
  }

  const tab = await findDashboardTab();
  if (tab) {
    const currentTabUrl = safeParseUrl(tab.url || "");
    const tabPath = currentTabUrl?.pathname || "/";
    const shouldOpenDashboardPage = tabPath === "/" || tabPath === "/home.html";

    if (shouldOpenDashboardPage) {
      await updateTab(tab.id, { url: dashboardUrl, active: true });
    } else {
      await updateTab(tab.id, { active: true });
    }

    if (selectedView !== "dashboard") {
      await sendTabMessage(tab.id, { action: "myam_project_set_member_view", view: selectedView });
      await reloadTab(tab.id);
    }
  } else {
    await createTab({ url: dashboardUrl });
  }

  window.close();
}

async function logoutDashboardSession() {
  setLoading(true, "Menghapus sesi...");

  const tab = await findDashboardTab();
  if (tab) {
    await sendTabMessage(tab.id, { action: "myam_project_clear_session" });
    await reloadTab(tab.id);
  }

  await clearStoredSession();
  renderDisconnected("Sesi dihapus. Login ulang di dashboard untuk menggunakan ekstensi.");
  setLoading(false);
  showToast("Logout berhasil.", "success");
}

async function bootstrapSession(manual = false) {
  setLoading(true, "Menyinkronkan sesi...");

  try {
    const synced = await syncSessionFromDashboardTab();
    const stored = await readStoredSession();

    let auth = null;
    if (synced) {
      if (synced.token) {
        auth = { token: synced.token, baseUrl: synced.baseUrl, user: synced.user };
      } else {
        await clearStoredSession();
      }
    } else if (stored.token) {
      auth = stored;
    }

    if (!auth || !auth.token) {
      renderDisconnected("Login di website MYAM Project lalu klik Sinkron Ulang.");
      return;
    }

    state.baseUrl = auth.baseUrl || DEFAULT_BASE_URL;
    state.token = auth.token;

    const sessionPayload = await fetchSessionData(state.baseUrl, state.token);

    const currentVersion = chrome.runtime.getManifest().version;
    const minVersion = sessionPayload.minExtensionVersion;
    if (isVersionOlder(currentVersion, minVersion)) {
      renderUpdateRequired(minVersion);
      return;
    }

    // Normalize user data to store
    const userToStore = sessionPayload.user || {
      email: sessionPayload.email,
      name: sessionPayload.name,
      role: sessionPayload.role,
      isPremium: sessionPayload.isPremium
    };

    await storageSet({
      [EXT_KEYS.token]: state.token,
      [EXT_KEYS.baseUrl]: state.baseUrl,
      [EXT_KEYS.user]: JSON.stringify(userToStore)
    });

    renderSessionInfo(sessionPayload, auth);


  } catch (error) {
    await clearStoredSession();
    if (error.message === "SUSPENDED") {
      renderSuspended(error.suspendMessage, error.adminContact);
    } else {
      const message = getFriendlyError(error.message);
      renderDisconnected(message);
      if (manual) {
        showToast(message, "error");
      }
    }
  } finally {
    setLoading(false);
  }
}

function bindEvents() {
  els.refreshBtn?.addEventListener("click", () => bootstrapSession(true));
  els.retrySyncBtn?.addEventListener("click", () => bootstrapSession(true));
  els.openDashboardBtn?.addEventListener("click", () => openDashboard("dashboard"));
  els.logoutBtn?.addEventListener("click", logoutDashboardSession);

  els.serviceSearch?.addEventListener("input", () => renderServices());

  const quickLinks = document.querySelectorAll(".quick-link");
  quickLinks.forEach((button) => {
    button.addEventListener("click", () => {
      const requestedView = button.getAttribute("data-view") || "dashboard";
      openDashboard(requestedView);
    });
  });
}

document.addEventListener("DOMContentLoaded", () => {
  els.statusIndicator = $("status-indicator");
  els.statusText = $("status-text");
  els.statusHint = $("status-hint");

  els.authCard = $("auth-card");
  els.openDashboardBtn = $("open-dashboard-btn");
  els.retrySyncBtn = $("retry-sync-btn");

  els.userCard = $("user-card");
  els.userName = $("user-name");
  els.userEmail = $("user-email");
  els.memberBadge = $("member-badge");

  els.controlsCard = $("controls-card");
  els.serviceSearch = $("service-search");

  els.servicesCard = $("services-card");
  els.serviceList = $("service-list");
  els.servicesEmpty = $("services-empty");
  els.servicesCount = $("services-count");

  els.logoutBtn = $("logout-btn");
  els.refreshBtn = $("refresh-btn");
  els.themeToggle = $("theme-toggle");
  els.loadingOverlay = $("loading-overlay");
  els.loadingText = $("loading-text");
  els.toast = $("toast");

  applyTheme(localStorage.getItem(THEME_KEY) || "light");
  els.themeToggle?.addEventListener("click", toggleTheme);

  // Dengarkan perubahan storage dari background.js (misal: forced logout)
  chrome.storage.onChanged.addListener((changes, namespace) => {
    if (namespace === "local" && changes[EXT_KEYS.token]) {
      if (!changes[EXT_KEYS.token].newValue) {
        // Token dihapus oleh background.js
        state.token = "";
        renderDisconnected("Sesi Anda telah berakhir atau perangkat mencapai batas. Silakan login kembali di website.");
        showToast("Sesi berakhir", "error");
      }
    }
  });

  bindEvents();
  bootstrapSession(false);
});
